import sys, numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileProgram, compileShader

# ---------- Shaders ----------
VERTEX_SRC = """
#version 330
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main() {
    texCoord = (pos + 1.0) * 0.5;
    gl_Position = vec4(pos,0,1);
}
"""

FRAGMENT_SRC = """
#version 330
in vec2 texCoord;
out vec4 fragColor;

uniform sampler2D latticeTex;
uniform float cycle;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeHeight;
uniform int yOffset;

float hdgl_slot(float val, float r_dim, float omega, int x, int y){
    float resonance = (x % 4 == 0 ? 0.1 * sin(float(cycle)*0.05 + float(y)) : 0.0);
    float wave = (x % 3 == 0 ? 0.3 : (x % 3 == 1 ? 0.0 : -0.3));
    float omega_inst = phiPowers[y % 72];
    float rec = r_dim*val*0.5 + 0.25*sin(float(cycle)*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val > threshold ? 1.0 : 0.0;
}

void main(){
    int x = int(texCoord.x * 32.0);
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    float val = texelFetch(latticeTex, ivec2(x,y - yOffset), 0).r;
    float r_dim = 0.3 + 0.01*float(y);
    float new_val = hdgl_slot(val, r_dim, sin(omegaTime), x, y);
    fragColor = vec4(new_val, sin(omegaTime), 0.0, 1.0);
}
"""

# ---------- Globals ----------
window = None
shader = None
vao = None
textures = []
fbos = []
current = 0
cycle = 0.0
omega_time = 0.0

# ---------- Lattice parameters ----------
lattice_width = 32
num_instances = 150_000         # Reduced from 15M for stability
max_tex_height = 2048           # Safe texture height
sub_tile_height = 256           # Smaller sub-tile per frame
threshold = np.sqrt(1.6180339887)
phi = 1.6180339887
phi_powers = np.array([1.0 / pow(phi, 7*(i+1)) for i in range(72)], dtype=np.float32)

# ---------- Tile division ----------
tile_count = (num_instances + max_tex_height - 1) // max_tex_height
tile_heights = [min(max_tex_height, num_instances - i*max_tex_height) for i in range(tile_count)]

# ---------- OpenGL Init ----------
def init_gl():
    global shader, vao, textures, fbos
    shader = compileProgram(compileShader(VERTEX_SRC, GL_VERTEX_SHADER),
                            compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER))
    verts = np.array([-1,-1,1,-1,-1,1,1,-1,1,1,-1,1], dtype=np.float32)
    vao = glGenVertexArrays(1)
    glBindVertexArray(vao)
    vbo = glGenBuffers(1)
    glBindBuffer(GL_ARRAY_BUFFER, vbo)
    glBufferData(GL_ARRAY_BUFFER, verts.nbytes, verts, GL_STATIC_DRAW)
    glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,None)
    glEnableVertexAttribArray(0)

    for t, th in enumerate(tile_heights):
        tex_pair = glGenTextures(2)
        fbo_pair = glGenFramebuffers(2)
        for i in range(2):
            glBindTexture(GL_TEXTURE_2D, tex_pair[i])
            glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, lattice_width, th, 0, GL_RGBA, GL_FLOAT, None)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)
            glBindFramebuffer(GL_FRAMEBUFFER, fbo_pair[i])
            glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, tex_pair[i],0)
        textures.append(tex_pair)
        fbos.append(fbo_pair)
        # Initialize lattice with zeros
        for start in range(0, th, sub_tile_height):
            end = min(start + sub_tile_height, th)   # <- use th directly
            init_chunk = np.zeros((end-start, lattice_width, 4), dtype=np.float32)
            glBindTexture(GL_TEXTURE_2D, tex_pair[0])
            glTexSubImage2D(GL_TEXTURE_2D, 0, 0, start, lattice_width, end-start, GL_RGBA, GL_FLOAT, init_chunk)


    glBindFramebuffer(GL_FRAMEBUFFER,0)

# ---------- Display ----------
def display():
    global cycle, omega_time, current
    next_idx = 1-current
    for t, th in enumerate(tile_heights):
        for y_start in range(0, th, sub_tile_height):
            h = min(sub_tile_height, th - y_start)
            glBindFramebuffer(GL_FRAMEBUFFER, fbos[t][next_idx])
            glViewport(0,0,lattice_width,h)
            glUseProgram(shader)
            glUniform1i(glGetUniformLocation(shader,"latticeTex"),0)
            glUniform1f(glGetUniformLocation(shader,"cycle"),cycle)
            glUniform1f(glGetUniformLocation(shader,"omegaTime"),omega_time)
            glUniform1fv(glGetUniformLocation(shader,"phiPowers"),72,phi_powers)
            glUniform1f(glGetUniformLocation(shader,"threshold"),threshold)
            glUniform1i(glGetUniformLocation(shader,"latticeHeight"), h)
            glUniform1i(glGetUniformLocation(shader,"yOffset"), y_start)
            glActiveTexture(GL_TEXTURE0)
            glBindTexture(GL_TEXTURE_2D, textures[t][current])
            glBindVertexArray(vao)
            glDrawArrays(GL_TRIANGLES,0,6)

    glBindFramebuffer(GL_FRAMEBUFFER,0)
    glViewport(0,0,1280,720)
    for t, th in enumerate(tile_heights):
        glBindTexture(GL_TEXTURE_2D, textures[t][next_idx])
        glDrawArrays(GL_TRIANGLES,0,6)

    glutSwapBuffers()
    cycle += 1
    omega_time += 0.05
    current = next_idx

def idle():
    glutPostRedisplay()

def main():
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL 150k Instance GPU Lattice + PHI Clock")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutMainLoop()

if __name__=="__main__":
    main()
